<?php
require_once __DIR__ . '/../config/db.php';
require_once __DIR__ . '/message_logging.php';

function sendWhatsApp($phoneNumber, $message) {
    global $pdo;
    try {
        // Get WhatsApp settings from database
        $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'whatsapp_settings'");
        $stmt->execute();
        $settings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

        // Check if WhatsApp is enabled
        if (($settings['enable_whatsapp'] ?? '0') !== '1') {
            logMessageAttempt('whatsapp', $phoneNumber, $message, 'failed', 'WhatsApp notifications are not enabled');
            throw new Exception('WhatsApp notifications are not enabled');
        }

        // Get WhatsApp method preference
        $whatsappMethod = $settings['whatsapp_method'] ?? 'web_link'; // web_link, api, or business_app

        // Format phone number (remove spaces, ensure it starts with country code)
        $phoneNumber = preg_replace('/\s+/', '', $phoneNumber);
        if (!str_starts_with($phoneNumber, '+')) {
            $phoneNumber = '+' . $phoneNumber;
        }

        // Remove + and any non-numeric characters for WhatsApp
        $cleanPhone = preg_replace('/[^0-9]/', '', $phoneNumber);

        switch ($whatsappMethod) {
            case 'api':
                return sendViaWhatsAppAPI($phoneNumber, $message, $settings);
            case 'business_app':
                return sendViaWhatsAppBusiness($phoneNumber, $message, $settings);
            case 'web_link':
            default:
                return sendViaWhatsAppWebLink($cleanPhone, $message, $settings);
        }
    } catch (Exception $e) {
        logMessageAttempt('whatsapp', $phoneNumber, $message, 'failed', $e->getMessage());
        error_log("WhatsApp Error: " . $e->getMessage());
        throw $e;
    }
}

function sendViaWhatsAppWebLink($phoneNumber, $message, $settings) {
    // Create multiple WhatsApp URLs for better compatibility
    $encodedMessage = urlencode($message);
    
    // Primary URL - wa.me (works on most devices)
    $whatsappUrl = "https://wa.me/{$phoneNumber}?text={$encodedMessage}";
    
    // Alternative URLs for better app detection
    $alternativeUrls = [
        'app_url' => "whatsapp://send?phone={$phoneNumber}&text={$encodedMessage}",
        'web_url' => "https://web.whatsapp.com/send?phone={$phoneNumber}&text={$encodedMessage}",
        'api_url' => "https://api.whatsapp.com/send?phone={$phoneNumber}&text={$encodedMessage}"
    ];
    
    // Log the attempt
    logMessageAttempt('whatsapp', $phoneNumber, $message, 'success', '', 'WhatsApp Web link generated');
    
    // Return the WhatsApp link for manual sending or automatic opening
    return [
        'success' => true,
        'method' => 'web_link',
        'whatsapp_url' => $whatsappUrl,
        'alternative_urls' => $alternativeUrls,
        'message' => 'WhatsApp link generated successfully. Click to open WhatsApp with pre-filled message.'
    ];
}

function sendViaWhatsAppBusiness($phoneNumber, $message, $settings) {
    // This method uses WhatsApp Business app integration
    // You can integrate with WhatsApp Business app on a phone/tablet
    
    $businessPhone = $settings['whatsapp_business_phone'] ?? '';
    if (empty($businessPhone)) {
        throw new Exception('WhatsApp Business phone number not configured');
    }
    
    // Create multiple WhatsApp URLs for better compatibility
    $encodedMessage = urlencode($message);
    
    // Primary URL - wa.me (works on most devices)
    $whatsappUrl = "https://wa.me/{$phoneNumber}?text={$encodedMessage}";
    
    // Alternative URLs for better app detection
    $alternativeUrls = [
        'app_url' => "whatsapp://send?phone={$phoneNumber}&text={$encodedMessage}",
        'web_url' => "https://web.whatsapp.com/send?phone={$phoneNumber}&text={$encodedMessage}",
        'api_url' => "https://api.whatsapp.com/send?phone={$phoneNumber}&text={$encodedMessage}"
    ];
    
    logMessageAttempt('whatsapp', $phoneNumber, $message, 'success', '', 'WhatsApp Business link generated');
    
    return [
        'success' => true,
        'method' => 'business_app',
        'whatsapp_url' => $whatsappUrl,
        'alternative_urls' => $alternativeUrls,
        'message' => 'WhatsApp Business link generated successfully.'
    ];
}

function sendViaWhatsAppAPI($phoneNumber, $message, $settings) {
    // Get WhatsApp API settings
    $apiKey = $settings['whatsapp_api_key'] ?? '';
    $apiUrl = $settings['whatsapp_api_url'] ?? '';
    $phoneNumberId = $settings['whatsapp_phone_number_id'] ?? '';

    // Validate required settings
    if (empty($apiKey) || empty($apiUrl) || empty($phoneNumberId)) {
        logMessageAttempt('whatsapp', $phoneNumber, $message, 'failed', 'WhatsApp API settings are not properly configured');
        throw new Exception('WhatsApp API settings are not properly configured');
    }

    $url = $apiUrl . "/{$phoneNumberId}/messages";
    
    $data = [
        'messaging_product' => 'whatsapp',
        'to' => $phoneNumber,
        'type' => 'text',
        'text' => [
            'body' => $message
        ]
    ];

    $headers = [
        'Authorization: Bearer ' . $apiKey,
        'Content-Type: application/json'
    ];

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);

    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($httpCode !== 200) {
        throw new Exception('Failed to send WhatsApp message: ' . $response);
    }

    logMessageAttempt('whatsapp', $phoneNumber, $message, 'success', '', 'Sent via WhatsApp API');
    return ['success' => true, 'method' => 'api', 'message' => 'Message sent via WhatsApp API'];
}

// Function to generate WhatsApp link for manual sending
function generateWhatsAppLink($phoneNumber, $message) {
    // Format phone number
    $phoneNumber = preg_replace('/\s+/', '', $phoneNumber);
    if (!str_starts_with($phoneNumber, '+')) {
        $phoneNumber = '+' . $phoneNumber;
    }
    
    // Remove + and any non-numeric characters
    $cleanPhone = preg_replace('/[^0-9]/', '', $phoneNumber);
    
    // Create WhatsApp Web link
    return "https://wa.me/{$cleanPhone}?text=" . urlencode($message);
}

// Function to test WhatsApp configuration
function testWhatsAppSettings() {
    global $pdo;
    try {
        // Get WhatsApp settings
        $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'whatsapp_settings'");
        $stmt->execute();
        $settings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

        if (($settings['enable_whatsapp'] ?? '0') !== '1') {
            logMessageAttempt('whatsapp', 'test', 'Test WhatsApp', 'failed', 'WhatsApp notifications are not enabled');
            return ['success' => false, 'message' => 'WhatsApp notifications are not enabled'];
        }

        $whatsappMethod = $settings['whatsapp_method'] ?? 'web_link';
        $testNumber = $settings['whatsapp_test_number'] ?? '';

        if (empty($testNumber)) {
            logMessageAttempt('whatsapp', 'test', 'Test WhatsApp', 'failed', 'Test phone number not configured');
            return ['success' => false, 'message' => 'Test phone number not configured'];
        }

        // Send a test message
        $testMessage = "This is a test WhatsApp message from your POS system. If you receive this, your WhatsApp configuration is working correctly.";
        
        $result = sendWhatsApp($testNumber, $testMessage);
        
        if ($result['success']) {
            logMessageAttempt('whatsapp', $testNumber, $testMessage, 'success', '', 'Test message sent successfully');
            return [
                'success' => true, 
                'message' => 'Test message sent successfully via ' . $result['method'],
                'whatsapp_url' => $result['whatsapp_url'] ?? null
            ];
        } else {
            return $result;
        }

    } catch (Exception $e) {
        logMessageAttempt('whatsapp', 'test', 'Test WhatsApp', 'failed', $e->getMessage());
        return ['success' => false, 'message' => $e->getMessage()];
    }
}

// Function to get WhatsApp status and configuration
function getWhatsAppStatus() {
    global $pdo;
    try {
        $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'whatsapp_settings'");
        $stmt->execute();
        $settings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

        return [
            'enabled' => ($settings['enable_whatsapp'] ?? '0') === '1',
            'method' => $settings['whatsapp_method'] ?? 'web_link',
            'configured' => !empty($settings['whatsapp_test_number'] ?? ''),
            'settings' => $settings
        ];
    } catch (Exception $e) {
        return [
            'enabled' => false,
            'method' => 'web_link',
            'configured' => false,
            'error' => $e->getMessage()
        ];
    }
} 